/**
  @file  main.c
  @brief main entry of the BLE stack sample application.

  <!--
  Copyright 2013 - 2015 Texas Instruments Incorporated. All rights reserved.

  IMPORTANT: Your use of this Software is limited to those specific rights
  granted under the terms of a software license agreement between the user
  who downloaded the software, his/her employer (which must be your employer)
  and Texas Instruments Incorporated (the "License").  You may not use this
  Software unless you agree to abide by the terms of the License. The License
  limits your use, and you acknowledge, that the Software may not be modified,
  copied or distributed unless embedded on a Texas Instruments microcontroller
  or used solely and exclusively in conjunction with a Texas Instruments radio
  frequency transceiver, which is integrated into your product.  Other than for
  the foregoing purpose, you may not use, reproduce, copy, prepare derivative
  works of, modify, distribute, perform, display or sell this Software and/or
  its documentation for any purpose.

  YOU FURTHER ACKNOWLEDGE AND AGREE THAT THE SOFTWARE AND DOCUMENTATION ARE
  PROVIDED ``AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESS OR IMPLIED,
  INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY, TITLE,
  NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT SHALL
  TEXAS INSTRUMENTS OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER CONTRACT,
  NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR OTHER
  LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
  INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE
  OR CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT
  OF SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
  (INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.

  Should you have any questions regarding your right to use this Software,
  contact Texas Instruments Incorporated at www.TI.com.
  -->
*/
#include <xdc/runtime/Error.h>

#include <ti/sysbios/family/arm/cc26xx/Power.h>
#include <ti/sysbios/BIOS.h>

#include "ICall.h"
#include "bcomdef.h"
#include "peripheral.h"
#include "simpleBLEPeripheral.h"
#include <stdio.h>

/* Header files required to enable instruction fetch cache */
#include <inc/hw_memmap.h>
#include <driverlib/vims.h>

//Begin copied code
#include "Board.h"
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/family/arm/m3/Hwi.h>

#include <ti/drivers/PIN/PINCC26XX.h>

#include <driverlib/aux_adc.h>
#include <driverlib/aux_wuc.h>
#include <inc/hw_aux_evctl.h>

#define TASK_STACK_SIZE 512
#define TASK_PRI        1


char taskStack[TASK_STACK_SIZE];
Task_Struct taskStruct;

Semaphore_Struct sem;
Semaphore_Handle hSem;

Hwi_Struct hwi;

#define SAMPLECOUNT 8
#define SAMPLETYPE uint16_t
#define SAMPLESIZE sizeof(SAMPLETYPE)

#define Board_TEMP_SENS   IOID_10

SAMPLETYPE adcSamples[SAMPLECOUNT];
SAMPLETYPE singleSample;

// Analog light sensor pins
const PIN_Config temp_config_table[] = {
		Board_LED3 		| PIN_GPIO_OUTPUT_EN | PIN_GPIO_HIGH | PIN_PUSHPULL | PIN_DRVSTR_MAX,
		Board_TEMP_SENS  | PIN_INPUT_DIS	  | PIN_GPIO_OUTPUT_DIS 	  | PIN_NOPULL,    			 //Temp Sensor configured as input and floating
		PIN_TERMINATE
};

PIN_Handle pinHandle;
PIN_State  pinState;

int j = 0;
double led_blink_time = 0;
double sample_avg = 0;
double sample_sum = 0;

void taskFxn(UArg a0, UArg a1);
void adcIsr(UArg a0);
//End copied code

#ifndef USE_DEFAULT_USER_CFG

#include "bleUserConfig.h"

// BLE user defined configuration
bleUserCfg_t user0Cfg = BLE_USER_CFG;

#endif // USE_DEFAULT_USER_CFG

/**
 * Exception handler
 */
void exceptionHandler()
{
	volatile uint8_t i = 1;
    while(i){}
}

#ifdef FEATURE_OAD
#if defined(__IAR_SYSTEMS_ICC__)
extern uint32_t __vector_table;
#elif defined (__TI_COMPILER_VERSION__)
extern uint32_t ti_sysbios_family_arm_m3_Hwi_resetVectors;
#endif //Compiler
#endif //FEATURE_OAD

/*
 *  ======== main ========
 */
int main()
{
  PIN_init(BoardGpioInitTable);

//Begin copied code
  //Initialize task
  Task_Params params;
  Task_Params_init(&params);
  params.priority = TASK_PRI;
  params.stackSize = TASK_STACK_SIZE;
  params.stack = taskStack;

  Task_construct(&taskStruct, taskFxn, &params, NULL);

  // Construct semaphore used for pending in task
  Semaphore_Params sParams;
  Semaphore_Params_init(&sParams);
  sParams.mode = Semaphore_Mode_BINARY;

  Semaphore_construct(&sem, 0, &sParams);
  hSem = Semaphore_handle(&sem);
//End copied code

#ifndef POWER_SAVING
    /* Set constraints for Standby, powerdown and idle mode */
    Power_setConstraint(Power_SB_DISALLOW);
    Power_setConstraint(Power_IDLE_PD_DISALLOW);
#endif // POWER_SAVING
    
    /* Initialize ICall module */
    ICall_init();

    /* Start tasks of external images - Priority 5 */
    ICall_createRemoteTasks();
    
    /* Kick off profile - Priority 3 */
    GAPRole_createTask();
    
    SimpleBLEPeripheral_createTask();

#ifdef FEATURE_OAD
    {
      uint8_t counter;
      uint32_t *vectorTable =  (uint32_t*) 0x20000000;
#if defined(__IAR_SYSTEMS_ICC__)
      uint32_t *flashVectors = &__vector_table;
#elif defined(__TI_COMPILER_VERSION__)
      uint32_t *flashVectors = &ti_sysbios_family_arm_m3_Hwi_resetVectors;
#endif //Compiler.
      
      // Write image specific interrupt vectors into RAM vector table.
      for(counter = 0; counter < 15; ++counter)
      {
        *vectorTable++ = *flashVectors++;
      }
    }
#endif //FEATURE_OAD
    
    /* enable interrupts and start SYS/BIOS */
    BIOS_start();
    
    return 0;
}

//Start copied code
void taskFxn(UArg a0, UArg a1) {

  Hwi_Params hwiParams;
  Hwi_Params_init(&hwiParams);
  hwiParams.enableInt = true;

  Hwi_construct(&hwi, INT_AUX_ADC, adcIsr, &hwiParams, NULL);

  //UART_Params uParams;
  // Initialize default values
  //UART_Params_init(&uParams);
  // Configure custom data, don't care about read params as not used
  // 115.2kBaud, Text, blocking mode
  //uHandle = UART_open(Board_UART,&uParams);

  // Set up pins
  pinHandle = PIN_open(&pinState, temp_config_table);

  // Enable clock for ADC digital and analog interface (not currently enabled in driver)
  AUXWUCClockEnable(AUX_WUC_ADI_CLOCK|AUX_WUC_SOC_CLOCK);
  // Connect AUX IO7 (DIO23) as analog input. Light sensor on SmartRF06EB
  AUXADCSelectInput(ADC_COMPB_IN_AUXIO5);


  // Set up ADC
  AUXADCEnableSync(AUXADC_REF_FIXED, AUXADC_SAMPLE_TIME_2P7_US, AUXADC_TRIGGER_MANUAL);

  // Disallow STANDBY mode while using the ADC.
  Power_setConstraint(Power_SB_DISALLOW);

  uint8_t currentSample = 0;

  while(currentSample < SAMPLECOUNT) {

    //Sleep 100ms in IDLE mode
    Task_sleep(100 * 1000 / Clock_tickPeriod);

    // Trigger ADC sampling
    AUXADCGenManualTrigger();
    // Wait in IDLE until done
    Semaphore_pend(hSem, BIOS_WAIT_FOREVER );

    adcSamples[currentSample++] = singleSample;

  }

  // Disable ADC
  AUXADCDisable();
  // Allow STANDBY mode again
  Power_releaseConstraint(Power_SB_DISALLOW);



  sample_sum = 0;
  for(j = 0; j < SAMPLECOUNT; j++)
  {
	  sample_sum += adcSamples[j];
  }
  sample_avg = sample_sum / SAMPLECOUNT;

  // Log data through LED
  led_blink_time = sample_avg*10;
  	for(j = 0; j < led_blink_time; j++)
  	{
  		PIN_setOutputValue(pinHandle, Board_LED3, 1);
  		Task_sleep(0.3 * 100000); //Pause for 0.5 second
  		PIN_setOutputValue(pinHandle, Board_LED3, 0);
  		Task_sleep(0.3 * 100000); //Pause for 0.5 second
  	}



    // Restore pins to values in BoardGpioTable
    PIN_close(pinHandle);

    // Log data through UART
    //for(uint8_t i = 0; i<SAMPLECOUNT; i++ ) {
    //  UART_write(uHandle, &adcSamples[i], SAMPLESIZE);

    // Goto STANDBY forever
    Task_sleep(BIOS_WAIT_FOREVER);

  }

void adcIsr(UArg a0) {

  // Pop sample from FIFO to allow clearing ADC_IRQ event
  singleSample = AUXADCReadFifo();
  // Clear ADC_IRQ flag. Note: Missing driver for this.
  HWREGBITW(AUX_EVCTL_BASE + AUX_EVCTL_O_EVTOMCUFLAGSCLR, AUX_EVCTL_EVTOMCUFLAGSCLR_ADC_IRQ_BITN) = 1;

  // Post semaphore to wakeup task
  Semaphore_post(hSem);

}
//End copied code

/**
 * Error handled to be hooked into TI-RTOS
 */
Void smallErrorHook(Error_Block *eb)
{
  for (;;);
}

/**
 * HAL assert handler required by OSAL memory module.
 */
void halAssertHandler(void)
{
  for (;;);
}
